<?php
/**
 * Nestict M-Pesa Callback Handler for STK Push
 * Place at: /modules/gateways/callback/nestictmpesa.php
 *
 * This script expects to be invoked by Safaricom Daraja POSTing JSON.
 * It will:
 * - parse the stkCallback structure
 * - find matching tblmpesarequests row (by MerchantRequestID or CheckoutRequestID) to get invoice_id
 * - extract MpesaReceiptNumber and Amount and apply addInvoicePayment() if not already applied
 * - log everything via logModuleCall()
 */

if (!defined('WHMCS')) {
    // If not loaded inside WHMCS environment, try to bootstrap it.
    require_once __DIR__ . '/../../init.php';
}

use WHMCS\Database\Capsule;

// Read raw body
$raw = file_get_contents('php://input');
logModuleCall('nestictmpesa', 'callback_raw', [], $raw);

$data = json_decode($raw, true);
if (!$data) {
    header('HTTP/1.1 400 Bad Request');
    echo 'Invalid JSON';
    exit;
}

// Daraja sends: Body->stkCallback
$stkCallback = $data['Body']['stkCallback'] ?? null;
if (!$stkCallback) {
    header('HTTP/1.1 400 Bad Request');
    echo 'No stkCallback';
    exit;
}

$resultCode = $stkCallback['ResultCode'] ?? null;
$merchantRequestID = $stkCallback['MerchantRequestID'] ?? null;
$checkoutRequestID = $stkCallback['CheckoutRequestID'] ?? null;

// Find stored mapping
$requestRow = null;
try {
    if (class_exists('WHMCS\\Database\\Capsule')) {
        $requestRow = Capsule::table('tblmpesarequests')
            ->where('merchant_request_id', $merchantRequestID)
            ->orWhere('checkout_request_id', $checkoutRequestID)
            ->orderBy('id', 'desc')
            ->first();
    }
} catch (Exception $e) {
    logModuleCall('nestictmpesa', 'callback_db_error', ['merchantRequest' => $merchantRequestID, 'checkoutRequest' => $checkoutRequestID], $e->getMessage());
}

$invoiceId = 0;
if ($requestRow && isset($requestRow->invoice_id)) {
    $invoiceId = (int)$requestRow->invoice_id;
}

// If no mapping found, attempt to read AccountReference from CallbackMetadata
if (!$invoiceId) {
    if (isset($stkCallback['CallbackMetadata']['Item']) && is_array($stkCallback['CallbackMetadata']['Item'])) {
        foreach ($stkCallback['CallbackMetadata']['Item'] as $item) {
            if (isset($item['Name']) && in_array($item['Name'], ['AccountReference', 'Account Ref', 'Account'])) {
                $accountRef = $item['Value'];
                // attempt to resolve invoice by invoice_num or invoicenum
                try {
                    $row = Capsule::table('tblinvoices')->select('id')->where('invoice_num', $accountRef)->orWhere('invoicenum', $accountRef)->first();
                    if ($row) {
                        $invoiceId = (int)$row->id;
                        break;
                    }
                } catch (Exception $e) {
                    logModuleCall('nestictmpesa', 'resolve_invoice_error', ['accountRef' => $accountRef], $e->getMessage());
                }
            }
        }
    }
}

// If result code indicates success, extract amount & receipt and apply payment
if ((int)$resultCode === 0) {
    $amount = null;
    $receipt = null;
    $phone = null;
    if (isset($stkCallback['CallbackMetadata']['Item']) && is_array($stkCallback['CallbackMetadata']['Item'])) {
        foreach ($stkCallback['CallbackMetadata']['Item'] as $item) {
            if (isset($item['Name'])) {
                if ($item['Name'] === 'Amount') $amount = $item['Value'];
                if (in_array($item['Name'], ['MpesaReceiptNumber', 'ReceiptNo'])) $receipt = $item['Value'];
                if (in_array($item['Name'], ['PhoneNumber', 'PartyA'])) $phone = $item['Value'];
            }
        }
    }

    // fallback to other possible fields
    if (!$receipt && isset($stkCallback['MpesaReceiptNumber'])) $receipt = $stkCallback['MpesaReceiptNumber'];

    if ($invoiceId && $amount && $receipt) {
        try {
            // Duplicate check
            $dup = Capsule::table('tblaccounts')->where('transid', $receipt)->count();
            if ($dup == 0) {
                addInvoicePayment($invoiceId, $receipt, $amount, 0, 'nestictmpesa');
                logModuleCall('nestictmpesa', 'callback_applied', ['invoice' => $invoiceId, 'receipt' => $receipt, 'amount' => $amount], 'success');
                // update request status if mapping present
                if ($requestRow && isset($requestRow->id)) {
                    try {
                        Capsule::table('tblmpesarequests')->where('id', $requestRow->id)->update([
                            'status' => 'Success',
                            'response' => json_encode($stkCallback),
                        ]);
                    } catch (Exception $e) {
                        logModuleCall('nestictmpesa', 'update_request_status_error', ['id' => $requestRow->id], $e->getMessage());
                    }
                }
            } else {
                logModuleCall('nestictmpesa', 'callback_duplicate', ['receipt' => $receipt, 'invoice' => $invoiceId], 'duplicate');
            }
        } catch (Exception $e) {
            logModuleCall('nestictmpesa', 'callback_apply_error', ['exception' => $e->getMessage(), 'data' => $stkCallback], 'failure');
        }
    } else {
        logModuleCall('nestictmpesa', 'callback_missing_info', ['invoice' => $invoiceId, 'amount' => $amount, 'data' => $stkCallback], 'missing');
    }

    header('HTTP/1.1 200 OK');
    echo 'OK';
    exit;
} else {
    // Failed or cancelled
    logModuleCall('nestictmpesa', 'callback_failed', $stkCallback, 'failure');
    // update request status to failed when possible
    if ($requestRow && isset($requestRow->id)) {
        try {
            Capsule::table('tblmpesarequests')->where('id', $requestRow->id)->update([
                'status' => 'Failed',
                'response' => json_encode($stkCallback),
            ]);
        } catch (Exception $e) {
            logModuleCall('nestictmpesa', 'update_request_status_error', ['id' => $requestRow->id], $e->getMessage());
        }
    }
    header('HTTP/1.1 200 OK');
    echo 'Callback received - failure';
    exit;
}