<?php
session_start();

// Enable error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Define the config file path
$config_path = 'config.php';

// Verify config file exists
if (!file_exists($config_path)) {
    die('Error: Config file not found at ' . htmlspecialchars($config_path));
}

require_once $config_path;

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || !isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    header('Location: /codes/login.php');
    exit;
}

// Initialize variables
$search_query = $_GET['search'] ?? '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 10;
$offset = ($page - 1) * $perPage;
$success_message = '';
$error_message = '';
$results = [];
$debug_log = [];
$log_file = '/home/nestict5/scripts.nestict.net/codes/debug.log';

try {
    // Verify PDO connection
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        die('Error: PDO connection not initialized in config.php. Please check your database configuration.');
    }

    // Test database connection
    $pdo->query('SELECT 1');
    $debug_log[] = "Database connection successful";

    // Handle add campus
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_campus'])) {
        $name = trim($_POST['name'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $county = trim($_POST['county'] ?? '');

        if ($name && $location && $county) {
            $sql = "INSERT INTO kmtc_campuses (name, location, county) 
                    VALUES (:name, :location, :county)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                'name' => $name,
                'location' => $location,
                'county' => $county
            ]);
            $success_message = "KMTC campus added successfully!";
            $debug_log[] = "Campus added: $name, $location, $county";
        } else {
            $error_message = "Please fill in all fields.";
        }
    }

    // Handle update campus
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_campus'])) {
        $id = (int)($_POST['id'] ?? 0);
        $name = trim($_POST['name'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $county = trim($_POST['county'] ?? '');

        if ($id && $name && $location && $county) {
            $sql = "UPDATE kmtc_campuses SET name = :name, location = :location, county = :county WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                'id' => $id,
                'name' => $name,
                'location' => $location,
                'county' => $county
            ]);
            $success_message = "KMTC campus updated successfully!";
            $debug_log[] = "Campus updated: ID $id, $name, $location, $county";
        } else {
            $error_message = "Please fill in all fields.";
        }
    }

    // Handle delete campus
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_campus'])) {
        $id = (int)($_POST['id'] ?? 0);
        if ($id) {
            $sql = "DELETE FROM kmtc_campuses WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(['id' => $id]);
            $success_message = "KMTC campus deleted successfully!";
            $debug_log[] = "Campus deleted: ID $id";
        } else {
            $error_message = "Invalid campus ID.";
        }
    }

    // Handle export requests
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export'])) {
        $exportType = $_POST['export'];
        $sql = "SELECT id, name, location, county FROM kmtc_campuses 
                WHERE LOWER(name) LIKE LOWER(:search) 
                OR LOWER(location) LIKE LOWER(:search) 
                OR LOWER(county) LIKE LOWER(:search)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['search' => "%$search_query%"]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if ($exportType === 'pdf') {
            if (!file_exists('vendor/autoload.php')) {
                die("PDF export requires dompdf. Please install via Composer.");
            }
            require 'vendor/autoload.php';
            $dompdf = new Dompdf\Dompdf();
            $html = '
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    table { width: 100%; border-collapse: collapse; }
                    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                    th { background-color: #1e88e5; color: white; }
                    h2 { color: #1e88e5; }
                </style>
            </head>
            <body>
                <h2>KMTC Campuses</h2>
                <table>
                    <tr><th>ID</th><th>Name</th><th>Location</th><th>County</th></tr>';
            foreach ($results as $row) {
                $html .= '<tr><td>' . htmlspecialchars($row['id']) . '</td><td>' . htmlspecialchars($row['name']) . '</td><td>' . htmlspecialchars($row['location']) . '</td><td>' . htmlspecialchars($row['county']) . '</td></tr>';
            }
            $html .= '</table></body></html>';
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', ' portrait');
            $dompdf->render();
            $dompdf->stream('kmtc_campuses.pdf', ['Attachment' => true]);
            exit;
        } elseif ($exportType === 'excel') {
            if (!file_exists('vendor/autoload.php')) {
                die("Excel export requires phpspreadsheet. Please install via Composer.");
            }
            require 'vendor/autoload.php';
            $spreadsheet = new PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setCellValue('A1', 'ID');
            $sheet->setCellValue('B1', 'Name');
            $sheet->setCellValue('C1', 'Location');
            $sheet->setCellValue('D1', 'County');
            $rowNumber = 2;
            foreach ($results as $row) {
                $sheet->setCellValue('A' . $rowNumber, $row['id']);
                $sheet->setCellValue('B' . $rowNumber, $row['name']);
                $sheet->setCellValue('C' . $rowNumber, $row['location']);
                $sheet->setCellValue('D' . $rowNumber, $row['county']);
                $rowNumber++;
            }
            $writer = new PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="kmtc_campuses.xlsx"');
            header('Cache-Control: max-age=0');
            $writer->save('php://output');
            exit;
        } elseif ($exportType === 'word') {
            header('Content-Type: application/vnd.openxmlformats-officedocument.wordprocessingml.document');
            header('Content-Disposition: attachment;filename="kmtc_campuses.docx"');
            header('Cache-Control: max-age=0');
            $xml = '<?xml version="1.0" encoding="UTF-8"?>
                    <w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
                        <w:body>
                            <w:p><w:r><w:t>KMTC Campuses</w:t></w:r></w:p>
                            <w:tbl>
                                <w:tr><w:tc><w:p><w:r><w:t>ID</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Name</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Location</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>County</w:t></w:r></w:p></w:tc></w:tr>';
            foreach ($results as $row) {
                $xml .= '<w:tr><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['id']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['name']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['location']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['county']) . '</w:t></w:r></w:p></w:tc></w:tr>';
            }
            $xml .= '</w:tbl></w:body></w:document>';
            echo $xml;
            exit;
        } elseif ($exportType === 'txt') {
            header('Content-Type: text/plain');
            header('Content-Disposition: attachment;filename="kmtc_campuses.txt"');
            header('Cache-Control: max-age=0');
            $txt = "KMTC Campuses\n\nID\tName\tLocation\tCounty\n";
            foreach ($results as $row) {
                $txt .= "{$row['id']}\t{$row['name']}\t{$row['location']}\t{$row['county']}\n manoeuvre;
            }
            echo $txt;
            exit;
        }
    }

    // Count total records for pagination
    $countSql = "SELECT COUNT(*) FROM kmtc_campuses 
                 WHERE LOWER(name) LIKE LOWER(:search) 
                 OR LOWER(location) LIKE LOWER(:search) 
                 OR LOWER(county) LIKE LOWER(:search)";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute(['search' => "%$search_query%"]);
    $totalRecords = $countStmt->fetchColumn();
    $totalPages = ceil($totalRecords / $perPage);

    // Fetch paginated results
    $sql = "SELECT id, name, location, county 
            FROM kmtc_campuses 
            WHERE LOWER(name) LIKE LOWER(:search) 
            OR LOWER(location) LIKE LOWER(:search) 
            OR LOWER(county) LIKE LOWER(:search) 
            ORDER BY name ASC 
            LIMIT :perPage OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':search', "%$search_query%", PDO::PARAM_STR);
    $stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $debug_log[] = "Query executed: SELECT ... WHERE search = '$search_query', page = $page, offset = $offset, results = " . count($results);

    // Write debug log
    file_put_contents($log_file, implode("\n", $debug_log) . "\n", FILE_APPEND);

} catch (PDOException $e) {
    $error_message = "Database error: " . $e->getMessage();
    file_put_contents($log_file, "Error: " . $e->getMessage() . "\n", FILE_APPEND);
} catch (Exception $e) {
    $error_message = "General error: " . $e->getMessage();
    file_put_contents($log_file, "Error: " . $e->getMessage() . "\n", FILE_APPEND);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>KMTC Admin Portal</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        :root {
            --primary-color: #1e88e5;
            --bg-color: #f8f9fa;
            --card-bg: #ffffff;
            --text-color: #212529;
            --border-radius: 12px;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
            --success-color: #2ecc71;
            --error-color: #e74c3c;
        }

        [data-theme="dark"] {
            --bg-color: #1a1a1a;
            --card-bg: #2c2c2c;
            --text-color: #e0e0e0;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
        }

        body {
            font-family: 'Inter', -apple-system, sans-serif;
            background-color: var(--bg-color);
            color: var(--text-color);
            margin: 0;
            padding: 80px 20px 40px;
            min-height: 100vh;
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            align-items: center;
        }

        .navbar {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background-color: var(--card-bg);
            box-shadow: var(--shadow);
            padding: 15px 20px;
            z-index: 1000;
        }

        .navbar-container {
            max-width: 1000px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .navbar-brand {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--primary-color);
            text-decoration: none;
        }

        .nav-menu {
            display: flex;
            gap: 20px;
        }

        .nav-menu a {
            color: var(--text-color);
            text-decoration: none;
            font-size: 1rem;
            font-weight: 500;
            transition: var(--transition);
        }

        .nav-menu a:hover {
            color: var(--primary-color);
        }

        .hamburger {
            display: none;
            flex-direction: column;
            cursor: pointer;
            background: none;
            border: none;
            padding: 5px;
        }

        .hamburger span {
            width: 25px;
            height: 3px;
            background-color: var(--text-color);
            margin: 2px 0;
            transition: var(--transition);
        }

        .theme-toggle {
            position: fixed;
            top: 20px;
            right: 20px;
            background: none;
            border: none;
            font-size