<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Define the config file path
$config_path = 'config.php';

// Verify config file exists
if (!file_exists($config_path)) {
    die('Error: Config file not found at ' . htmlspecialchars($config_path));
}

require_once $config_path;

// Initialize variables
$search_query = $_GET['search'] ?? '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 10;
$offset = ($page - 1) * $perPage;
$success_message = '';
$error_message = '';
$results = [];
$debug_log = [];
$log_file = '/home/nestict5/scripts.nestict.net/codes/debug.log';

try {
    // Verify PDO connection
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        die('Error: PDO connection not initialized in config.php. Please check your database configuration.');
    }

    // Test database connection
    $pdo->query('SELECT 1');
    $debug_log[] = "Database connection successful";

    // Handle export requests
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export'])) {
        $exportType = $_POST['export'];
        $sql = "SELECT institution_name, knecode, region, county, subcounty, institution_type, gender, category, accommodation_type 
                FROM schools 
                WHERE LOWER(institution_name) LIKE LOWER(:search) 
                OR LOWER(region) LIKE LOWER(:search) 
                OR LOWER(county) LIKE LOWER(:search) 
                OR LOWER(subcounty) LIKE LOWER(:search) 
                OR LOWER(institution_type) LIKE LOWER(:search) 
                OR LOWER(gender) LIKE LOWER(:search) 
                OR LOWER(category) LIKE LOWER(:search) 
                OR LOWER(accommodation_type) LIKE LOWER(:search)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['search' => "%$search_query%"]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if ($exportType === 'pdf') {
            if (!file_exists('vendor/autoload.php')) {
                die("PDF export requires dompdf. Please install via Composer.");
            }
            require 'vendor/autoload.php';
            $dompdf = new Dompdf\Dompdf();
            $html = '
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    table { width: 100%; border-collapse: collapse; }
                    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                    th { background-color: #1e88e5; color: white; }
                    h2 { color: #1e88e5; }
                </style>
            </head>
            <body>
                <h2>School Directory</h2>
                <table>
                    <tr><th>Institution Name</th><th>KNEC Code</th><th>Region</th><th>County</th><th>Sub-County</th><th>Institution Type</th><th>Gender</th><th>Category</th><th>Accommodation Type</th></tr>';
            foreach ($results as $row) {
                $html .= '<tr><td>' . htmlspecialchars($row['institution_name']) . '</td><td>' . htmlspecialchars($row['knecode']) . '</td><td>' . htmlspecialchars($row['region']) . '</td><td>' . htmlspecialchars($row['county']) . '</td><td>' . htmlspecialchars($row['subcounty']) . '</td><td>' . htmlspecialchars($row['institution_type']) . '</td><td>' . htmlspecialchars($row['gender']) . '</td><td>' . htmlspecialchars($row['category']) . '</td><td>' . htmlspecialchars($row['accommodation_type']) . '</td></tr>';
            }
            $html .= '</table></body></html>';
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();
            $dompdf->stream('schools_directory.pdf', ['Attachment' => true]);
            exit;
        } elseif ($exportType === 'excel') {
            if (!file_exists('vendor/autoload.php')) {
                die("Excel export requires phpspreadsheet. Please install via Composer.");
            }
            require 'vendor/autoload.php';
            $spreadsheet = new PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setCellValue('A1', 'Institution Name');
            $sheet->setCellValue('B1', 'KNEC Code');
            $sheet->setCellValue('C1', 'Region');
            $sheet->setCellValue('D1', 'County');
            $sheet->setCellValue('E1', 'Sub-County');
            $sheet->setCellValue('F1', 'Institution Type');
            $sheet->setCellValue('G1', 'Gender');
            $sheet->setCellValue('H1', 'Category');
            $sheet->setCellValue('I1', 'Accommodation Type');
            $rowNumber = 2;
            foreach ($results as $row) {
                $sheet->setCellValue('A' . $rowNumber, $row['institution_name']);
                $sheet->setCellValue('B' . $rowNumber, $row['knecode']);
                $sheet->setCellValue('C' . $rowNumber, $row['region']);
                $sheet->setCellValue('D' . $rowNumber, $row['county']);
                $sheet->setCellValue('E' . $rowNumber, $row['subcounty']);
                $sheet->setCellValue('F' . $rowNumber, $row['institution_type']);
                $sheet->setCellValue('G' . $rowNumber, $row['gender']);
                $sheet->setCellValue('H' . $rowNumber, $row['category']);
                $sheet->setCellValue('I' . $rowNumber, $row['accommodation_type']);
                $rowNumber++;
            }
            $writer = new PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="schools_directory.xlsx"');
            header('Cache-Control: max-age=0');
            $writer->save('php://output');
            exit;
        } elseif ($exportType === 'word') {
            header('Content-Type: application/vnd.openxmlformats-officedocument.wordprocessingml.document');
            header('Content-Disposition: attachment;filename="schools_directory.docx"');
            header('Cache-Control: max-age=0');
            $xml = '<?xml version="1.0" encoding="UTF-8"?>
                    <w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
                        <w:body>
                            <w:p><w:r><w:t>School Directory</w:t></w:r></w:p>
                            <w:tbl>
                                <w:tr><w:tc><w:p><w:r><w:t>Institution Name</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>KNEC Code</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Region</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>County</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Sub-County</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Institution Type</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Gender</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Category</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Accommodation Type</w:t></w:r></w:p></w:tc></w:tr>';
            foreach ($results as $row) {
                $xml .= '<w:tr><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['institution_name']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['knecode']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['region']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['county']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['subcounty']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['institution_type']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['gender']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['category']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['accommodation_type']) . '</w:t></w:r></w:p></w:tc></w:tr>';
            }
            $xml .= '</w:tbl></w:body></w:document>';
            echo $xml;
            exit;
        } elseif ($exportType === 'txt') {
            header('Content-Type: text/plain');
            header('Content-Disposition: attachment;filename="schools_directory.txt"');
            header('Cache-Control: max-age=0');
            $txt = "School Directory\n\nInstitution Name\tKNEC Code\tRegion\tCounty\tSub-County\tInstitution Type\tGender\tCategory\tAccommodation Type\n";
            foreach ($results as $row) {
                $txt .= "{$row['institution_name']}\t{$row['knecode']}\t{$row['region']}\t{$row['county']}\t{$row['subcounty']}\t{$row['institution_type']}\t{$row['gender']}\t{$row['category']}\t{$row['accommodation_type']}\n";
            }
            echo $txt;
            exit;
        }
    }

    // Fetch unique regions for the initial dropdown
    $regionStmt = $pdo->query("SELECT DISTINCT region FROM schools ORDER BY region ASC");
    $regions = $regionStmt->fetchAll(PDO::FETCH_ASSOC);

    // Count total records for pagination (only if subcounty is selected)
    if (isset($_GET['subcounty']) && !empty($_GET['subcounty'])) {
        $subcounty = $_GET['subcounty'];
        $countSql = "SELECT COUNT(*) FROM schools WHERE subcounty = :subcounty";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute(['subcounty' => $subcounty]);
        $totalRecords = $countStmt->fetchColumn();
        $totalPages = ceil($totalRecords / $perPage);

        // Fetch paginated results based on subcounty
        $sql = "SELECT institution_name, knecode, region, county, subcounty, institution_type, gender, category, accommodation_type 
                FROM schools 
                WHERE subcounty = :subcounty 
                ORDER BY institution_name ASC 
                LIMIT :perPage OFFSET :offset";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':subcounty', $subcounty, PDO::PARAM_STR);
        $stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $debug_log[] = "Query executed: SELECT ... WHERE subcounty = '$subcounty', page = $page, offset = $offset, results = " . count($results);
    }

    // Write debug log
    file_put_contents($log_file, implode("\n", $debug_log) . "\n", FILE_APPEND);

} catch (PDOException $e) {
    $error_message = "Database error: " . $e->getMessage();
    file_put_contents($log_file, "Error: " . $e->getMessage() . "\n", FILE_APPEND);
} catch (Exception $e) {
    $error_message = "General error: " . $e->getMessage();
    file_put_contents($log_file, "Error: " . $e->getMessage() . "\n", FILE_APPEND);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>School Directory</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #2c3e50;
            --secondary-color: #3498db;
            --bg-color: #f5f7fa;
            --card-bg: #ffffff;
            --text-color: #2c3e50;
            --border-radius: 12px;
            --shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
            --success-color: #27ae60;
            --error-color: #e74c3c;
        }

        [data-theme="dark"] {
            --bg-color: #1a1e24;
            --card-bg: #2c2f36;
            --text-color: #ecf0f1;
            --shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, var(--bg-color) 0%, #e0e7f0 100%);
            color: var(--text-color);
            min-height: 100vh;
            padding: 80px 20px 40px;
            transition: var(--transition);
        }

        .navbar {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
            box-shadow: var(--shadow);
            padding: 15px 20px;
            z-index: 1000;
        }

        .navbar-container {
            max-width: 1200px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .navbar-brand {
            font-size: 1.8rem;
            font-weight: 700;
            color: #fff;
            text-decoration: none;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .nav-menu {
            display: flex;
            gap: 20px;
            align-items: center;
        }

        .nav-menu a {
            color: #fff;
            text-decoration: none;
            font-size: 1.1rem;
            font-weight: 500;
            transition: var(--transition);
            padding: 5px 10px;
            border-radius: 6px;
        }

        .nav-menu a:hover {
            background: rgba(255, 255, 255, 0.2);
            color: #fff;
        }

        .hamburger {
            display: none;
            flex-direction: column;
            cursor: pointer;
            background: none;
            border: none;
            padding: 5px;
        }

        .hamburger span {
            width: 25px;
            height: 3px;
            background: #fff;
            margin: 3px 0;
            transition: var(--transition);
        }

        .theme-toggle {
            position: fixed;
            top: 20px;
            right: 20px;
            background: rgba(255, 255, 255, 0.1);
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #fff;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1001;
            transition: var(--transition);
        }

        .theme-toggle:hover {
            background: rgba(255, 255, 255, 0.3);
        }

        h2 {
            color: var(--primary-color);
            margin-bottom: 25px;
            font-size: 2.5rem;
            font-weight: 700;
            text-align: center;
            text-transform: uppercase;
            letter-spacing: 2px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            width: 100%;
        }

        .filter-section {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            justify-content: center;
            margin-bottom: 30px;
        }

        .filter-section select {
            padding: 12px;
            font-size: 1rem;
            border: none;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            background: var(--card-bg);
            color: var(--text-color);
            transition: var(--transition);
            width: 200px;
        }

        .filter-section select:focus {
            outline: none;
            box-shadow: 0 0 10px rgba(52, 152, 219, 0.5);
        }

        .search-info {
            margin-bottom: 20px;
            font-size: 1.2rem;
            font-weight: 500;
            color: var(--text-color);
            text-align: center;
            background: var(--card-bg);
            padding: 15px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .search-info span {
            color: var(--secondary-color);
            font-weight: 600;
        }

        .search-form {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 15px;
            width: 100%;
            max-width: 700px;
            margin-bottom: 30px;
        }

        .search-container {
            position: relative;
            width: 100%;
        }

        input[type="text"] {
            padding: 15px 50px 15px 20px;
            width: 100%;
            font-size: 1.1rem;
            border: none;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            background: var(--card-bg);
            color: var(--text-color);
            transition: var(--transition);
        }

        input[type="text"]:focus {
            outline: none;
            box-shadow: 0 0 10px rgba(52, 152, 219, 0.5);
        }

        .search-icon {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--secondary-color);
            font-size: 1.2rem;
        }

        .button-group {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            justify-content: center;
        }

        button {
            background: linear-gradient(45deg, var(--primary-color), var(--secondary-color));
            color: #fff;
            border: none;
            padding: 12px 30px;
            font-size: 1rem;
            font-weight: 600;
            border-radius: var(--border-radius);
            cursor: pointer;
            transition: var(--transition);
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        button:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow);
        }

        button.reset-btn {
            background: linear-gradient(45deg, #7f8c8d, #95a5a6);
        }

        button.clear-btn {
            background: linear-gradient(45deg, #c0392b, #e74c3c);
        }

        button.export-btn {
            background: linear-gradient(45deg, #219653, #27ae60);
        }

        button.print-btn {
            background: linear-gradient(45deg, #7d3c98, #8e44ad);
        }

        .message {
            text-align: center;
            margin-bottom: 20px;
            font-weight: 500;
            padding: 10px;
            border-radius: var(--border-radius);
        }

        .success {
            background: rgba(39, 174, 96, 0.1);
            color: var(--success-color);
        }

        .error {
            background: rgba(231, 76, 60, 0.1);
            color: var(--error-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background: var(--card-bg);
            border-radius: var(--border-radius);
            overflow: hidden;
            box-shadow: var(--shadow);
            margin-top: 20px;
            display: none;
        }

        table.visible {
            display: table;
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
        }

        th {
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
            color: #fff;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        tr {
            transition: var(--transition);
        }

        tr:hover {
            background: rgba(52, 152, 219, 0.05);
        }

        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
        }

        .pagination a {
            text-decoration: none;
            padding: 10px 20px;
            border-radius: var(--border-radius);
            background: var(--card-bg);
            color: var(--text-color);
            transition: var(--transition);
            border: 1px solid rgba(0, 0, 0, 0.1);
        }

        .pagination a:hover {
            background: var(--secondary-color);
            color: #fff;
            border-color: var(--secondary-color);
        }

        .pagination a.active {
            background: var(--primary-color);
            color: #fff;
            border-color: var(--primary-color);
            cursor: default;
        }

        .pagination a.disabled {
            color: #ccc;
            cursor: not-allowed;
            pointer-events: none;
        }

        .no-results {
            text-align: center;
            margin-top: 30px;
            font-size: 1.2rem;
            color: #7f8c8d;
            background: var(--card-bg);
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .spinner {
            display: none;
            border: 5px solid #f3f3f3;
            border-top: 5px solid var(--secondary-color);
            border-radius: 50%;
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        @media print {
            .navbar, .theme-toggle, .search-form, .pagination, .button-group, .filter-section {
                display: none;
            }
            body {
                padding: 20px;
                background: #fff;
            }
            table {
                display: table !important;
                box-shadow: none;
            }
            .container {
                max-width: 100%;
            }
            h2 {
                text-align: center;
            }
        }

        @media (max-width: 768px) {
            .nav-menu {
                display: none;
                flex-direction: column;
                position: absolute;
                top: 60px;
                left: 0;
                right: 0;
                background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
                padding: 20px;
                box-shadow: var(--shadow);
                border-radius: 0 0 var(--border-radius) var(--border-radius);
            }

            .nav-menu.active {
                display: flex;
            }

            .hamburger {
                display: flex;
            }

            .filter-section {
                flex-direction: column;
                align-items: center;
            }

            .filter-section select {
                width: 100%;
                max-width: 300px;
            }

            table, thead, tbody, th, td, tr {
                display: block;
            }

            thead {
                display: none;
            }

            tr {
                margin-bottom: 15px;
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
            }

            td {
                padding: 12px;
                position: relative;
                border: none;
                display: flex;
                justify-content: space-between;
                align-items: center;
                background: var(--card-bg);
            }

            td::before {
                content: attr(data-label);
                font-weight: 600;
                color: var(--primary-color);
                flex: 1;
                text-transform: uppercase;
                font-size: 0.9rem;
            }

            td::after {
                content: attr(data-value);
                flex: 2;
                text-align: right;
                font-size: 0.9rem;
            }

            h2 {
                font-size: 2rem;
            }

            .button-group {
                flex-direction: column;
                align-items: center;
            }

            button {
                width: 100%;
                margin-bottom: 10px;
            }
        }

        @media (max-width: 480px) {
            h2 {
                font-size: 1.8rem;
            }

            input[type="text"], .filter-section select, button {
                font-size: 1rem;
            }

            .navbar-brand {
                font-size: 1.5rem;
            }

            .pagination a {
                padding: 8px 15px;
                font-size: 0.9rem;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-container">
            <a href="/codes/jss.php" class="navbar-brand">School Directory</a>
            <button class="hamburger" onclick="toggleMenu()">
                <span></span>
                <span></span>
                <span></span>
            </button>
            <div class="nav-menu" id="nav-menu">
                <a href="/codes/search.php">Search Postal Codes</a>
                <a href="/codes/find.php">Search Banks</a>
                <a href="/about">About</a>
                <a href="/contact">Contact</a>
            </div>
        </div>
    </nav>
    <button class="theme-toggle" onclick="toggleTheme()">🌙</button>
    <div class="container">
        <h2>School Directory</h2>
        <?php if ($search_query): ?>
            <p class="search-info">Search results for: <span><?= htmlspecialchars($search_query) ?></span> (<?php echo $totalRecords; ?> results)</p>
        <?php endif; ?>
        <?php if ($success_message): ?>
            <p class="message success"><?= htmlspecialchars($success_message) ?></p>
        <?php endif; ?>
        <?php if ($error_message): ?>
            <p class="message error"><?= htmlspecialchars($error_message) ?></p>
        <?php endif; ?>
        <div class="filter-section">
            <select id="region" name="region">
                <option value="">Select Region</option>
                <?php foreach ($regions as $region): ?>
                    <option value="<?= htmlspecialchars($region['region']) ?>"><?= htmlspecialchars($region['region']) ?></option>
                <?php endforeach; ?>
            </select>
            <select id="county" name="county" disabled>
                <option value="">Select County</option>
            </select>
            <select id="subcounty" name="subcounty" disabled>
                <option value="">Select Sub-County</option>
            </select>
        </div>
        <form method="GET" class="search-form">
            <div class="search-container">
                <input type="text" id="search-input" name="search" placeholder="Search by Institution Name, Region, County, etc." 
                       value="<?= htmlspecialchars($search_query) ?>">
                <span class="search-icon">🔍</span>
            </div>
            <div class="button-group">
                <button type="submit">Search</button>
                <button type="button" class="reset-btn" onclick="resetSearch()">Reset</button>
                <button type="button" class="clear-btn" onclick="clearTable()">Clear Table</button>
            </div>
            <div id="loading" class="spinner"></div>
        </form>

        <?php if ($results): ?>
            <form method="POST" class="button-group">
                <button type="submit" name="export" value="pdf" class="export-btn">Export PDF</button>
                <button type="submit" name="export" value="excel" class="export-btn">Export Excel</button>
                <button type="submit" name="export" value="word" class="export-btn">Export Word</button>
                <button type="submit" name="export" value="txt" class="export-btn">Export TXT</button>
                <button type="button" class="print-btn" onclick="window.print()">Print</button>
            </form>
            <table id="results-table" class="visible">
                <thead>
                    <tr>
                        <th>Institution Name</th>
                        <th>KNEC Code</th>
                        <th>Region</th>
                        <th>County</th>
                        <th>Sub-County</th>
                        <th>Institution Type</th>
                        <th>Gender</th>
                        <th>Category</th>
                        <th>Accommodation Type</th>
                    </tr>
                </thead>
                <tbody id="results-body">
                    <?php foreach ($results as $row): ?>
                    <tr>
                        <td data-label="Institution Name" data-value="<?= htmlspecialchars($row['institution_name']) ?>">
                            <?= htmlspecialchars($row['institution_name']) ?>
                        </td>
                        <td data-label="KNEC Code" data-value="<?= htmlspecialchars($row['knecode']) ?>">
                            <?= htmlspecialchars($row['knecode']) ?>
                        </td>
                        <td data-label="Region" data-value="<?= htmlspecialchars($row['region']) ?>">
                            <?= htmlspecialchars($row['region']) ?>
                        </td>
                        <td data-label="County" data-value="<?= htmlspecialchars($row['county']) ?>">
                            <?= htmlspecialchars($row['county']) ?>
                        </td>
                        <td data-label="Sub-County" data-value="<?= htmlspecialchars($row['subcounty']) ?>">
                            <?= htmlspecialchars($row['subcounty']) ?>
                        </td>
                        <td data-label="Institution Type" data-value="<?= htmlspecialchars($row['institution_type']) ?>">
                            <?= htmlspecialchars($row['institution_type']) ?>
                        </td>
                        <td data-label="Gender" data-value="<?= htmlspecialchars($row['gender']) ?>">
                            <?= htmlspecialchars($row['gender']) ?>
                        </td>
                        <td data-label="Category" data-value="<?= htmlspecialchars($row['category']) ?>">
                            <?= htmlspecialchars($row['category']) ?>
                        </td>
                        <td data-label="Accommodation Type" data-value="<?= htmlspecialchars($row['accommodation_type']) ?>">
                            <?= htmlspecialchars($row['accommodation_type']) ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php
                $queryString = $search_query ? "&search=" . urlencode($search_query) : "";
                ?>
                <a href="?page=1&subcounty=<?= urlencode($subcounty) ?><?php echo $queryString; ?>" class="<?php echo $page <= 1 ? 'disabled' : ''; ?>">« First</a>
                <a href="?page=<?php echo $page - 1; ?>&subcounty=<?= urlencode($subcounty) ?><?php echo $queryString; ?>" class="<?php echo $page <= 1 ? 'disabled' : ''; ?>">‹ Prev</a>
                <?php
                $startPage = max(1, $page - 2);
                $endPage = min($totalPages, $page + 2);
                for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="?page=<?php echo $i; ?>&subcounty=<?= urlencode($subcounty) ?><?php echo $queryString; ?>" class="<?php echo $i === $page ? 'active' : ''; ?>"><?php echo $i; ?></a>
                <?php endfor; ?>
                <a href="?page=<?php echo $page + 1; ?>&subcounty=<?= urlencode($subcounty) ?><?php echo $queryString; ?>" class="<?php echo $page >= $totalPages ? 'disabled' : ''; ?>">Next ›</a>
                <a href="?page=<?php echo $totalPages; ?>&subcounty=<?= urlencode($subcounty) ?><?php echo $queryString; ?>" class="<?php echo $page >= $totalPages ? 'disabled' : ''; ?>">Last »</a>
            </div>
            <?php endif; ?>
        <?php else: ?>
            <p class="no-results">No results found.</p>
        <?php endif; ?>
    </div>

    <script>
        function toggleTheme() {
            const body = document.body;
            const currentTheme = body.getAttribute('data-theme');
            body.setAttribute('data-theme', currentTheme === 'dark' ? 'light' : 'dark');
            localStorage.setItem('theme', body.getAttribute('data-theme'));
        }

        function toggleMenu() {
            const menu = document.getElementById('nav-menu');
            menu.classList.toggle('active');
        }

        function resetSearch() {
            window.location.href = window.location.pathname;
        }

        function clearTable() {
            const table = document.getElementById('results-table');
            if (table) {
                table.classList.remove('visible');
            }
            $('#region, #county, #subcounty').prop('selectedIndex', 0).prop('disabled', false);
            $('#county, #subcounty').html('<option value="">Select County</option><option value="">Select Sub-County</option>');
        }

        $(document).ready(function() {
            const $region = $('#region');
            const $county = $('#county');
            const $subcounty = $('#subcounty');
            const $resultsBody = $('#results-body');
            const $resultsTable = $('#results-table');
            const $loading = $('#loading');

            // Load saved theme
            const savedTheme = localStorage.getItem('theme') || 'light';
            document.body.setAttribute('data-theme', savedTheme);

            // Load counties based on region
            $region.on('change', function() {
                const region = $(this).val();
                if (region) {
                    $.ajax({
                        url: 'ajax/get_counties.php',
                        method: 'POST',
                        data: { region: region },
                        dataType: 'json',
                        beforeSend: function() {
                            $loading.show();
                        },
                        success: function(response) {
                            $loading.hide();
                            if (response.error) {
                                alert(response.error);
                            } else {
                                $county.prop('disabled', false).html('<option value="">Select County</option>');
                                $.each(response.counties, function(index, county) {
                                    $county.append($('<option>', {
                                        value: county,
                                        text: county
                                    }));
                                });
                                $subcounty.prop('disabled', true).html('<option value="">Select Sub-County</option>');
                                $resultsTable.removeClass('visible');
                            }
                        },
                        error: function() {
                            $loading.hide();
                            alert('Failed to load counties. Please try again.');
                        }
                    });
                } else {
                    $county.prop('disabled', true).html('<option value="">Select County</option>');
                    $subcounty.prop('disabled', true).html('<option value="">Select Sub-County</option>');
                    $resultsTable.removeClass('visible');
                }
            });

            // Load subcounties based on county
            $county.on('change', function() {
                const county = $(this).val();
                const region = $region.val();
                if (county && region) {
                    $.ajax({
                        url: 'ajax/get_subcounties.php',
                        method: 'POST',
                        data: { county: county, region: region },
                        dataType: 'json',
                        beforeSend: function() {
                            $loading.show();
                        },
                        success: function(response) {
                            $loading.hide();
                            if (response.error) {
                                alert(response.error);
                            } else {
                                $subcounty.prop('disabled', false).html('<option value="">Select Sub-County</option>');
                                $.each(response.subcounties, function(index, subcounty) {
                                    $subcounty.append($('<option>', {
                                        value: subcounty,
                                        text: subcounty
                                    }));
                                });
                                $resultsTable.removeClass('visible');
                            }
                        },
                        error: function() {
                            $loading.hide();
                            alert('Failed to load sub-counties. Please try again.');
                        }
                    });
                } else {
                    $subcounty.prop('disabled', true).html('<option value="">Select Sub-County</option>');
                    $resultsTable.removeClass('visible');
                }
            });

            // Load schools based on subcounty
            $subcounty.on('change', function() {
                const subcounty = $(this).val();
                if (subcounty) {
                    $.ajax({
                        url: 'ajax/get_schools.php',
                        method: 'POST',
                        data: { subcounty: subcounty },
                        dataType: 'json',
                        beforeSend: function() {
                            $loading.show();
                        },
                        success: function(response) {
                            $loading.hide();
                            if (response.error) {
                                alert(response.error);
                            } else {
                                $resultsBody.empty();
                                if (response.schools.length > 0) {
                                    $.each(response.schools, function(index, school) {
                                        $resultsBody.append(`
                                            <tr>
                                                <td data-label="Institution Name" data-value="${school.institution_name}">${school.institution_name}</td>
                                                <td data-label="KNEC Code" data-value="${school.knecode}">${school.knecode}</td>
                                                <td data-label="Region" data-value="${school.region}">${school.region}</td>
                                                <td data-label="County" data-value="${school.county}">${school.county}</td>
                                                <td data-label="Sub-County" data-value="${school.subcounty}">${school.subcounty}</td>
                                                <td data-label="Institution Type" data-value="${school.institution_type}">${school.institution_type}</td>
                                                <td data-label="Gender" data-value="${school.gender}">${school.gender}</td>
                                                <td data-label="Category" data-value="${school.category}">${school.category}</td>
                                                <td data-label="Accommodation Type" data-value="${school.accommodation_type}">${school.accommodation_type}</td>
                                            </tr>
                                        `);
                                    });
                                    $resultsTable.addClass('visible');
                                } else {
                                    $resultsTable.removeClass('visible');
                                    $('<p>').addClass('no-results').text('No results found.').insertAfter('.search-form');
                                    setTimeout(() => $('.no-results').remove(), 3000);
                                }
                            }
                        },
                        error: function() {
                            $loading.hide();
                            alert('Failed to load schools. Please try again.');
                        }
                    });
                } else {
                    $resultsTable.removeClass('visible');
                }
            });

            // Live search with debounce
            const $searchInput = $('#search-input');
            let searchTimeout;
            $searchInput.on('input', function() {
                clearTimeout(searchTimeout);
                const query = $(this).val();
                searchTimeout = setTimeout(() => performSearch(query), 300);
            });

            function performSearch(query) {
                $loading.show();
                $.ajax({
                    url: 'schools_search.php',
                    method: 'POST',
                    data: { search_query: query },
                    dataType: 'json',
                    success: function(response) {
                        $loading.hide();
                        $resultsBody.empty();
                        if (response.error) {
                            $resultsTable.removeClass('visible');
                            $('<p>').addClass('message error').text(response.error).insertAfter('.search-form');
                            setTimeout(() => $('.message.error').remove(), 3000);
                        } else {
                            if (response.results.length > 0) {
                                response.results.forEach(row => {
                                    $resultsBody.append(`
                                        <tr>
                                            <td data-label="Institution Name" data-value="${row.institution_name}">${row.institution_name}</td>
                                            <td data-label="KNEC Code" data-value="${row.knecode}">${row.knecode}</td>
                                            <td data-label="Region" data-value="${row.region}">${row.region}</td>
                                            <td data-label="County" data-value="${row.county}">${row.county}</td>
                                            <td data-label="Sub-County" data-value="${row.subcounty}">${row.subcounty}</td>
                                            <td data-label="Institution Type" data-value="${row.institution_type}">${row.institution_type}</td>
                                            <td data-label="Gender" data-value="${row.gender}">${row.gender}</td>
                                            <td data-label="Category" data-value="${row.category}">${row.category}</td>
                                            <td data-label="Accommodation Type" data-value="${row.accommodation_type}">${row.accommodation_type}</td>
                                        </tr>
                                    `);
                                });
                                $resultsTable.addClass('visible');
                            } else {
                                $resultsTable.removeClass('visible');
                                $('<p>').addClass('no-results').text('No results found.').insertAfter('.search-form');
                                setTimeout(() => $('.no-results').remove(), 3000);
                            }
                        }
                    },
                    error: function() {
                        $loading.hide();
                        $resultsTable.removeClass('visible');
                        $('<p>').addClass('message error').text('Failed to fetch results. Please try again.').insertAfter('.search-form');
                        setTimeout(() => $('.message.error').remove(), 3000);
                    }
                });
            }
        });
    </script>
</body>
</html>