<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Define the config file path
$config_path = 'config.php';

// Verify config file exists
if (!file_exists($config_path)) {
    die('Error: Config file not found at ' . htmlspecialchars($config_path));
}

require_once $config_path;

// Initialize variables
$search_query = $_GET['search'] ?? '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 10;
$offset = ($page - 1) * $perPage;
$success_message = '';
$error_message = '';
$results = [];
$debug_log = [];
$log_file = '/home/nestict5/scripts.nestict.net/codes/debug.log';

try {
    // Verify PDO connection
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        die('Error: PDO connection not initialized in config.php. Please check your database configuration.');
    }

    // Test database connection
    $pdo->query('SELECT 1');
    $debug_log[] = "Database connection successful";

    // Handle add campus form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_campus'])) {
        $name = trim($_POST['name'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $county = trim($_POST['county'] ?? '');

        if ($name && $location && $county) {
            $sql = "INSERT INTO kmtc_campuses (name, location, county) 
                    VALUES (:name, :location, :county)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                'name' => $name,
                'location' => $location,
                'county' => $county
            ]);
            $success_message = "KMTC campus added successfully!";
            $debug_log[] = "Campus added: $name, $location, $county";
        } else {
            $error_message = "Please fill in all fields.";
        }
    }

    // Handle export requests
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export'])) {
        $exportType = $_POST['export'];
        $sql = "SELECT * FROM kmtc_campuses 
                WHERE LOWER(name) LIKE LOWER(:search) 
                OR LOWER(location) LIKE LOWER(:search) 
                OR LOWER(county) LIKE LOWER(:search)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['search' => "%$search_query%"]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if ($exportType === 'pdf') {
            if (!file_exists('vendor/autoload.php')) {
                die("PDF export requires dompdf. Please install via Composer.");
            }
            require 'vendor/autoload.php';
            $dompdf = new Dompdf\Dompdf();
            $html = '
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    table { width: 100%; border-collapse: collapse; }
                    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                    th { background-color: #1e88e5; color: white; }
                    h2 { color: #1e88e5; }
                </style>
            </head>
            <body>
                <h2>KMTC Campuses</h2>
                <table>
                    <tr><th>Name</th><th>Location</th><th>County</th></tr>';
            foreach ($results as $row) {
                $html .= '<tr><td>' . htmlspecialchars($row['name']) . '</td><td>' . htmlspecialchars($row['location']) . '</td><td>' . htmlspecialchars($row['county']) . '</td></tr>';
            }
            $html .= '</table></body></html>';
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();
            $dompdf->stream('kmtc_campuses.pdf', ['Attachment' => true]);
            exit;
        } elseif ($exportType === 'excel') {
            if (!file_exists('vendor/autoload.php')) {
                die("Excel export requires phpspreadsheet. Please install via Composer.");
            }
            require 'vendor/autoload.php';
            $spreadsheet = new PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setCellValue('A1', 'Name');
            $sheet->setCellValue('B1', 'Location');
            $sheet->setCellValue('C1', 'County');
            $rowNumber = 2;
            foreach ($results as $row) {
                $sheet->setCellValue('A' . $rowNumber, $row['name']);
                $sheet->setCellValue('B' . $rowNumber, $row['location']);
                $sheet->setCellValue('C' . $rowNumber, $row['county']);
                $rowNumber++;
            }
            $writer = new PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="kmtc_campuses.xlsx"');
            header('Cache-Control: max-age=0');
            $writer->save('php://output');
            exit;
        } elseif ($exportType === 'word') {
            header('Content-Type: application/vnd.openxmlformats-officedocument.wordprocessingml.document');
            header('Content-Disposition: attachment;filename="kmtc_campuses.docx"');
            header('Cache-Control: max-age=0');
            $xml = '<?xml version="1.0" encoding="UTF-8"?>
                    <w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
                        <w:body>
                            <w:p><w:r><w:t>KMTC Campuses</w:t></w:r></w:p>
                            <w:tbl>
                                <w:tr><w:tc><w:p><w:r><w:t>Name</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Location</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>County</w:t></w:r></w:p></w:tc></w:tr>';
            foreach ($results as $row) {
                $xml .= '<w:tr><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['name']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['location']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['county']) . '</w:t></w:r></w:p></w:tc></w:tr>';
            }
            $xml .= '</w:tbl></w:body></w:document>';
            echo $xml;
            exit;
        } elseif ($exportType === 'txt') {
            header('Content-Type: text/plain');
            header('Content-Disposition: attachment;filename="kmtc_campuses.txt"');
            header('Cache-Control: max-age=0');
            $txt = "KMTC Campuses\n\nName\tLocation\tCounty\n";
            foreach ($results as $row) {
                $txt .= "{$row['name']}\t{$row['location']}\t{$row['county']}\n";
            }
            echo $txt;
            exit;
        }
    }

    // Count total records for pagination
    $countSql = "SELECT COUNT(*) FROM kmtc_campuses 
                 WHERE LOWER(name) LIKE LOWER(:search) 
                 OR LOWER(location) LIKE LOWER(:search) 
                 OR LOWER(county) LIKE LOWER(:search)";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute(['search' => "%$search_query%"]);
    $totalRecords = $countStmt->fetchColumn();
    $totalPages = ceil($totalRecords / $perPage);

    // Fetch paginated results
    $sql = "SELECT name, location, county 
            FROM kmtc_campuses 
            WHERE LOWER(name) LIKE LOWER(:search) 
            OR LOWER(location) LIKE LOWER(:search) 
            OR LOWER(county) LIKE LOWER(:search) 
            ORDER BY name ASC 
            LIMIT :perPage OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':search', "%$search_query%", PDO::PARAM_STR);
    $stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $debug_log[] = "Query executed: SELECT ... WHERE search = '$search_query', page = $page, offset = $offset, results = " . count($results);

    // Write debug log
    file_put_contents($log_file, implode("\n", $debug_log) . "\n", FILE_APPEND);

} catch (PDOException $e) {
    $error_message = "Database error: " . $e->getMessage();
    file_put_contents($log_file, "Error: " . $e->getMessage() . "\n", FILE_APPEND);
} catch (Exception $e) {
    $error_message = "General error: " . $e->getMessage();
    file_put_contents($log_file, "Error: " . $e->getMessage() . "\n", FILE_APPEND);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>KMTC Campus Finder</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        :root {
            --primary-color: #1e88e5;
            --bg-color: #f8f9fa;
            --card-bg: #ffffff;
            --text-color: #212529;
            --border-radius: 12px;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
            --success-color: #2ecc71;
            --error-color: #e74c3c;
        }

        [data-theme="dark"] {
            --bg-color: #1a1a1a;
            --card-bg: #2c2c2c;
            --text-color: #e0e0e0;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
        }

        body {
            font-family: 'Inter', -apple-system, sans-serif;
            background-color: var(--bg-color);
            color: var(--text-color);
            margin: 0;
            padding: 80px 20px 40px;
            min-height: 100vh;
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            align-items: center;
        }

        .navbar {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background-color: var(--card-bg);
            box-shadow: var(--shadow);
            padding: 15px 20px;
            z-index: 1000;
        }

        .navbar-container {
            max-width: 1000px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .navbar-brand {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--primary-color);
            text-decoration: none;
        }

        .nav-menu {
            display: flex;
            gap: 20px;
        }

        .nav-menu a {
            color: var(--text-color);
            text-decoration: none;
            font-size: 1rem;
            font-weight: 500;
            transition: var(--transition);
        }

        .nav-menu a:hover {
            color: var(--primary-color);
        }

        .hamburger {
            display: none;
            flex-direction: column;
            cursor: pointer;
            background: none;
            border: none;
            padding: 5px;
        }

        .hamburger span {
            width: 25px;
            height: 3px;
            background-color: var(--text-color);
            margin: 2px 0;
            transition: var(--transition);
        }

        .theme-toggle {
            position: fixed;
            top: 20px;
            right: 20px;
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: var(--text-color);
            z-index: 1001;
        }

        h2 {
            color: var(--primary-color);
            margin-bottom: 20px;
            font-size: 2rem;
            font-weight: 600;
        }

        .container {
            width: 100%;
            max-width: 1000px;
            margin: 0 auto;
        }

        .search-info {
            margin-bottom: 20px;
            font-size: 1.1rem;
            font-weight: 500;
            color: var(--text-color);
            text-align: center;
        }

        .search-info span {
            color: var(--primary-color);
            font-weight: 600;
        }

        .search-form {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 15px;
            width: 100%;
            max-width: 600px;
            margin-bottom: 20px;
        }

        .search-container {
            position: relative;
            width: 100%;
        }

        input[type="text"] {
            padding: 12px 45px 12px 15px;
            width: 100%;
            font-size: 16px;
            border: 1px solid #ccc;
            border-radius: var(--border-radius);
            box-sizing: border-box;
            transition: var(--transition);
            background-color: var(--card-bg);
            color: var(--text-color);
        }

        input[type="text"]:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 8px rgba(30, 136, 229, 0.3);
        }

        .search-icon {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #888;
        }

        .button-group {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            justify-content: center;
        }

        button {
            background-color: var(--primary-color);
            color: white;
            border: none;
            padding: 12px 30px;
            font-size: 16px;
            font-weight: 500;
            border-radius: var(--border-radius);
            cursor: pointer;
            transition: var(--transition);
        }

        button:hover {
            background-color: #1565c0;
            transform: translateY(-2px);
        }

        button.reset-btn {
            background-color: #95a5a6;
        }

        button.reset-btn:hover {
            background-color: #7f8c8d;
        }

        button.clear-btn {
            background-color: #e74c3c;
        }

        button.clear-btn:hover {
            background-color: #c0392b;
        }

        button.export-btn {
            background-color: #27ae60;
        }

        button.export-btn:hover {
            background-color: #219653;
        }

        button.print-btn {
            background-color: #8e44ad;
        }

        button.print-btn:hover {
            background-color: #7d3c98;
        }

        .add-campus-form {
            background-color: var(--card-bg);
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            width: 100%;
            max-width: 600px;
            margin-bottom: 30px;
        }

        .add-campus-form h3 {
            margin-top: 0;
            color: var(--primary-color);
        }

        .add-campus-form .form-group {
            margin-bottom: 15px;
        }

        .add-campus-form label {
            display: block;
            font-weight: 500;
            margin-bottom: 5px;
        }

        .add-campus-form input {
            width: 100%;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: var(--border-radius);
            background-color: var(--card-bg);
            color: var(--text-color);
        }

        .message {
            text-align: center;
            margin-bottom: 20px;
            font-weight: 500;
        }

        .success {
            color: var(--success-color);
        }

        .error {
            color: var(--error-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            overflow: hidden;
            box-shadow: var(--shadow);
            margin-top: 20px;
            display: none;
        }

        table.visible {
            display: table;
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid rgba(0, 0, 0, 0.1);
        }

        th {
            background-color: var(--primary-color);
            color: white;
            font-weight: 600;
        }

        tr {
            transition: var(--transition);
        }

        tr:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }

        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
        }

        .pagination a {
            text-decoration: none;
            padding: 8px 16px;
            border-radius: var(--border-radius);
            background-color: var(--card-bg);
            color: var(--text-color);
            transition: var(--transition);
            border: 1px solid #ccc;
        }

        .pagination a:hover {
            background-color: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
        }

        .pagination a.active {
            background-color: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
            cursor: default;
        }

        .pagination a.disabled {
            color: #ccc;
            cursor: not-allowed;
            pointer-events: none;
        }

        .no-results {
            text-align: center;
            margin-top: 30px;
            font-size: 1.1rem;
            color: #666;
        }

        .spinner {
            display: none;
            border: 4px solid #f3f3f3;
            border-top: 4px solid #3498db;
            border-radius: 50%;
            width: 24px;
            height: 24px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        @media print {
            .navbar, .theme-toggle, .search-form, .add-campus-form, .pagination, .button-group {
                display: none;
            }
            body {
                padding: 20px;
                background: white;
            }
            table {
                display: table !important;
                box-shadow: none;
            }
            .container {
                max-width: 100%;
            }
            h2 {
                text-align: center;
            }
        }

        @media (max-width: 768px) {
            .nav-menu {
                display: none;
                flex-direction: column;
                position: absolute;
                top: 60px;
                left: 0;
                right: 0;
                background-color: var(--card-bg);
                padding: 20px;
                box-shadow: var(--shadow);
            }

            .nav-menu.active {
                display: flex;
            }

            .hamburger {
                display: flex;
            }

            table, thead, tbody, th, td, tr {
                display: block;
            }

            thead {
                display: none;
            }

            tr {
                margin-bottom: 10px;
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
            }

            td {
                padding: 12px;
                position: relative;
                border: none;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }

            td::before {
                content: attr(data-label);
                font-weight: 600;
                color: var(--primary-color);
                flex: 1;
            }

            td::after {
                content: attr(data-value);
                flex: 2;
                text-align: right;
            }
        }

        @media (max-width: 480px) {
            h2 {
                font-size: 1.5rem;
            }

            input[type="text"], button {
                font-size: 14px;
            }

            .navbar-brand {
                font-size: 1.2rem;
            }

            .button-group {
                flex-direction: column;
                align-items: center;
            }

            .pagination a {
                padding: 6px 12px;
                font-size: 14px;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-container">
            <a href="/codes/kmtc.php" class="navbar-brand">KMTC Campus Finder</a>
            <button class="hamburger" onclick="toggleMenu()">
                <span></span>
                <span></span>
                <span></span>
            </button>
            <div class="nav-menu" id="nav-menu">
                <a href="/codes/search.php">Search Postal Codes</a>
                <a href="/codes/find.php">Search Banks</a>
                <a href="/about">About</a>
                <a href="/contact">Contact</a>
            </div>
        </div>
    </nav>
    <button class="theme-toggle" onclick="toggleTheme()">🌙</button>
    <div class="container">
        <h2>KMTC Campus Finder</h2>
        <?php if ($search_query): ?>
            <p class="search-info">Search results for: <span><?= htmlspecialchars($search_query) ?></span> (<?php echo $totalRecords; ?> results)</p>
        <?php endif; ?>
        <?php if ($success_message): ?>
            <p class="message success"><?= htmlspecialchars($success_message) ?></p>
        <?php endif; ?>
        <?php if ($error_message): ?>
            <p class="message error"><?= htmlspecialchars($error_message) ?></p>
        <?php endif; ?>
        <form method="GET" class="search-form">
            <div class="search-container">
                <input type="text" id="search-input" name="search" placeholder="Search by campus name, location, or county" 
                       value="<?= htmlspecialchars($search_query) ?>">
                <span class="search-icon">🔍</span>
            </div>
            <div class="button-group">
                <button type="submit">Search</button>
                <button type="button" class="reset-btn" onclick="resetSearch()">Reset</button>
                <button type="button" class="clear-btn" onclick="clearTable()">Clear Table</button>
            </div>
            <div id="loading" class="spinner"></div>
        </form>

        <form method="POST" class="add-campus-form">
            <h3>Add New KMTC Campus</h3>
            <div class="form-group">
                <label for="name">Campus Name</label>
                <input type="text" name="name" id="name" placeholder="Enter campus name (e.g., KMTC Nairobi)">
            </div>
            <div class="form-group">
                <label for="location">Location</label>
                <input type="text" name="location" id="location" placeholder="Enter location (e.g., Nairobi)">
            </div>
            <div class="form-group">
                <label for="county">County</label>
                <input type="text" name="county" id="county" placeholder="Enter county (e.g., Nairobi)">
            </div>
            <button type="submit" name="add_campus">Add Campus</button>
        </form>

        <?php if ($results): ?>
            <form method="POST" class="button-group">
                <button type="submit" name="export" value="pdf" class="export-btn">Export PDF</button>
                <button type="submit" name="export" value="excel" class="export-btn">Export Excel</button>
                <button type="submit" name="export" value="word" class="export-btn">Export Word</button>
                <button type="submit" name="export" value="txt" class="export-btn">Export TXT</button>
                <button type="button" class="print-btn" onclick="window.print()">Print</button>
            </form>
            <table id="results-table" class="visible">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Location</th>
                        <th>County</th>
                    </tr>
                </thead>
                <tbody id="results-body">
                    <?php foreach ($results as $row): ?>
                    <tr>
                        <td data-label="Name" data-value="<?= htmlspecialchars($row['name']) ?>">
                            <?= htmlspecialchars($row['name']) ?>
                        </td>
                        <td data-label="Location" data-value="<?= htmlspecialchars($row['location']) ?>">
                            <?= htmlspecialchars($row['location']) ?>
                        </td>
                        <td data-label="County" data-value="<?= htmlspecialchars($row['county']) ?>">
                            <?= htmlspecialchars($row['county']) ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php
                $queryString = $search_query ? "&search=" . urlencode($search_query) : "";
                ?>
                <a href="?page=1<?php echo $queryString; ?>" class="<?php echo $page <= 1 ? 'disabled' : ''; ?>">« First</a>
                <a href="?page=<?php echo $page - 1; ?><?php echo $queryString; ?>" class="<?php echo $page <= 1 ? 'disabled' : ''; ?>">‹ Prev</a>
                <?php
                $startPage = max(1, $page - 2);
                $endPage = min($totalPages, $page + 2);
                for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="?page=<?php echo $i; ?><?php echo $queryString; ?>" class="<?php echo $i === $page ? 'active' : ''; ?>"><?php echo $i; ?></a>
                <?php endfor; ?>
                <a href="?page=<?php echo $page + 1; ?><?php echo $queryString; ?>" class="<?php echo $page >= $totalPages ? 'disabled' : ''; ?>">Next ›</a>
                <a href="?page=<?php echo $totalPages; ?><?php echo $queryString; ?>" class="<?php echo $page >= $totalPages ? 'disabled' : ''; ?>">Last »</a>
            </div>
            <?php endif; ?>
        <?php else: ?>
            <p class="no-results">No results found.</p>
        <?php endif; ?>
    </div>

    <script>
        function toggleTheme() {
            const body = document.body;
            const currentTheme = body.getAttribute('data-theme');
            body.setAttribute('data-theme', currentTheme === 'dark' ? 'light' : 'dark');
            localStorage.setItem('theme', body.getAttribute('data-theme'));
        }

        function toggleMenu() {
            const menu = document.getElementById('nav-menu');
            menu.classList.toggle('active');
        }

        function resetSearch() {
            window.location.href = window.location.pathname;
        }

        function clearTable() {
            const table = document.getElementById('results-table');
            if (table) {
                table.classList.remove('visible');
            }
        }

        $(document).ready(function() {
            const $searchInput = $('#search-input');
            const $resultsBody = $('#results-body');
            const $resultsTable = $('#results-table');
            const $loading = $('#loading');

            // Load saved theme
            const savedTheme = localStorage.getItem('theme') || 'light';
            document.body.setAttribute('data-theme', savedTheme);

            // Live search with debounce
            let searchTimeout;
            $searchInput.on('input', function() {
                clearTimeout(searchTimeout);
                const query = $(this).val();
                searchTimeout = setTimeout(() => performSearch(query), 300);
            });

            function performSearch(query) {
                $loading.show();
                $.ajax({
                    url: 'kmtc_search.php',
                    method: 'POST',
                    data: { search_query: query },
                    dataType: 'json',
                    success: function(response) {
                        $loading.hide();
                        $resultsBody.empty();
                        if (response.error) {
                            $resultsTable.removeClass('visible');
                            $('<p>').addClass('message error').text(response.error).insertAfter('.search-form');
                            setTimeout(() => $('.message.error').remove(), 3000);
                        } else {
                            if (response.results.length > 0) {
                                response.results.forEach(row => {
                                    $resultsBody.append(`
                                        <tr>
                                            <td data-label="Name" data-value="${row.name}">${row.name}</td>
                                            <td data-label="Location" data-value="${row.location}">${row.location}</td>
                                            <td data-label="County" data-value="${row.county}">${row.county}</td>
                                        </tr>
                                    `);
                                });
                                $resultsTable.addClass('visible');
                            } else {
                                $resultsTable.removeClass('visible');
                                $('<p>').addClass('no-results').text('No results found.').insertAfter('.search-form');
                                setTimeout(() => $('.no-results').remove(), 3000);
                            }
                        }
                    },
                    error: function() {
                        $loading.hide();
                        $resultsTable.removeClass('visible');
                        $('<p>').addClass('message error').text('Failed to fetch results. Please try again.').insertAfter('.search-form');
                        setTimeout(() => $('.message.error').remove(), 3000);
                    }
                });
            }
        });
    </script>
</body>
</html>