<?php
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

require 'config.php';
// Note: Install dependencies via Composer:
// composer require dompdf/dompdf
// composer require phpoffice/phpspreadsheet

use Dompdf\Dompdf;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

header('Content-Type: text/html; charset=utf-8');

$search = $_GET['search'] ?? '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 10;
$offset = ($page - 1) * $perPage;
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_code'])) {
    $code = trim($_POST['code'] ?? '');
    $town = trim($_POST['town'] ?? '');
    $county = trim($_POST['county'] ?? '');

    if ($code && $town && $county) {
        try {
            $sql = "INSERT INTO postal_codes (code, town, county) VALUES (:code, :town, :county)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(['code' => $code, 'town' => $town, 'county' => $county]);
            $success_message = "Postal code added successfully!";
        } catch (PDOException $e) {
            $error_message = "Error adding postal code: " . $e->getMessage();
        }
    } else {
        $error_message = "Please fill in all fields.";
    }
}

// Handle export requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export'])) {
    $exportType = $_POST['export'];
    $sql = "SELECT * FROM postal_codes 
            WHERE code LIKE :search 
            OR town LIKE :search 
            OR county LIKE :search 
            ORDER BY town ASC";
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['search' => "%$search%"]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Export query failed: " . $e->getMessage());
        die("Database error during export. Please try again later.");
    }

    if ($exportType === 'pdf') {
        if (!file_exists('vendor/autoload.php')) {
            die("PDF export requires dompdf. Please install via Composer.");
        }
        require 'vendor/autoload.php';
        $dompdf = new Dompdf();
        $html = '
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; }
                table { width: 100%; border-collapse: collapse; }
                th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                th { background-color: #1e88e5; color: white; }
                h2 { color: #1e88e5; }
            </style>
        </head>
        <body>
            <h2>Kenya Postal Codes</h2>
            <table>
                <tr><th>Postal Code</th><th>Town</th><th>County</th></tr>';
        foreach ($results as $row) {
            $html .= '<tr><td>' . htmlspecialchars($row['code']) . '</td><td>' . htmlspecialchars($row['town']) . '</td><td>' . htmlspecialchars($row['county']) . '</td></tr>';
        }
        $html .= '</table></body></html>';
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();
        $dompdf->stream('postal_codes.pdf', ['Attachment' => true]);
        exit;
    } elseif ($exportType === 'excel') {
        if (!file_exists('vendor/autoload.php')) {
            die("Excel export requires phpspreadsheet. Please install via Composer.");
        }
        require 'vendor/autoload.php';
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setCellValue('A1', 'Postal Code');
        $sheet->setCellValue('B1', 'Town');
        $sheet->setCellValue('C1', 'County');
        $rowNumber = 2;
        foreach ($results as $row) {
            $sheet->setCellValue('A' . $rowNumber, $row['code']);
            $sheet->setCellValue('B' . $rowNumber, $row['town']);
            $sheet->setCellValue('C' . $rowNumber, $row['county']);
            $rowNumber++;
        }
        $writer = new Xlsx($spreadsheet);
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="postal_codes.xlsx"');
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    } elseif ($exportType === 'word') {
        header('Content-Type: application/vnd.openxmlformats-officedocument.wordprocessingml.document');
        header('Content-Disposition: attachment;filename="postal_codes.docx"');
        header('Cache-Control: max-age=0');
        $xml = '<?xml version="1.0" encoding="UTF-8"?>
                <w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
                    <w:body>
                        <w:p><w:r><w:t>Kenya Postal Codes</w:t></w:r></w:p>
                        <w:tbl>
                            <w:tr><w:tc><w:p><w:r><w:t>Postal Code</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>Town</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>County</w:t></w:r></w:p></w:tc></w:tr>';
        foreach ($results as $row) {
            $xml .= '<w:tr><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['code']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['town']) . '</w:t></w:r></w:p></w:tc><w:tc><w:p><w:r><w:t>' . htmlspecialchars($row['county']) . '</w:t></w:r></w:p></w:tc></w:tr>';
        }
        $xml .= '</w:tbl></w:body></w:document>';
        echo $xml;
        exit;
    } elseif ($exportType === 'txt') {
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment;filename="postal_codes.txt"');
        header('Cache-Control: max-age=0');
        $txt = "Kenya Postal Codes\n\nPostal Code\tTown\tCounty\n";
        foreach ($results as $row) {
            $txt .= "{$row['code']}\t{$row['town']}\t{$row['county']}\n";
        }
        echo $txt;
        exit;
    }
}

// Count total records for pagination
$countSql = "SELECT COUNT(*) FROM postal_codes 
             WHERE code LIKE :search 
             OR town LIKE :search 
             OR county LIKE :search";
try {
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute(['search' => "%$search%"]);
    $totalRecords = $countStmt->fetchColumn();
    $totalPages = ceil($totalRecords / $perPage);
} catch (PDOException $e) {
    error_log("Count query failed: " . $e->getMessage());
    die("Database error. Please try again later.");
}

// Fetch paginated results
$sql = "SELECT * FROM postal_codes 
        WHERE code LIKE :search 
        OR town LIKE :search 
        OR county LIKE :search 
        ORDER BY town ASC 
        LIMIT :perPage OFFSET :offset";
try {
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
    $stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Data query failed: " . $e->getMessage());
    die("Database error. Please try again later.");
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Kenya Postal Code Search</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        :root {
            --primary-color: #1e88e5;
            --bg-color: #f8f9fa;
            --card-bg: #ffffff;
            --text-color: #212529;
            --border-radius: 12px;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
            --success-color: #2ecc71;
            --error-color: #e74c3c;
        }

        [data-theme="dark"] {
            --bg-color: #1a1a1a;
            --card-bg: #2c2c2c;
            --text-color: #e0e0e0;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
        }

        body {
            font-family: 'Inter', -apple-system, sans-serif;
            background-color: var(--bg-color);
            color: var(--text-color);
            margin: 0;
            padding: 80px 20px 40px;
            min-height: 100vh;
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            align-items: center;
        }

        .navbar {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background-color: var(--card-bg);
            box-shadow: var(--shadow);
            padding: 15px 20px;
            z-index: 1000;
        }

        .navbar-container {
            max-width: 1000px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .navbar-brand {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--primary-color);
            text-decoration: none;
        }

        .nav-menu {
            display: flex;
            gap: 20px;
        }

        .nav-menu a {
            color: var(--text-color);
            text-decoration: none;
            font-size: 1rem;
            font-weight: 500;
            transition: var(--transition);
        }

        .nav-menu a:hover {
            color: var(--primary-color);
        }

        .hamburger {
            display: none;
            flex-direction: column;
            cursor: pointer;
            background: none;
            border: none;
            padding: 5px;
        }

        .hamburger span {
            width: 25px;
            height: 3px;
            background-color: var(--text-color);
            margin: 2px 0;
            transition: var(--transition);
        }

        .theme-toggle {
            position: fixed;
            top: 20px;
            right: 20px;
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: var(--text-color);
            z-index: 1001;
        }

        h2 {
            color: var(--primary-color);
            margin-bottom: 20px;
            font-size: 2rem;
            font-weight: 600;
        }

        .container {
            width: 100%;
            max-width: 1000px;
            margin: 0 auto;
        }

        .search-info {
            margin-bottom: 20px;
            font-size: 1.1rem;
            font-weight: 500;
            color: var(--text-color);
            text-align: center;
        }

        .search-info span {
            color: var(--primary-color);
            font-weight: 600;
        }

        form.search-form {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 15px;
            width: 100%;
            max-width: 600px;
            margin-bottom: 20px;
        }

        .search-container {
            position: relative;
            width: 100%;
        }

        input[type="text"] {
            padding: 12px 45px 12px 15px;
            width: 100%;
            font-size: 16px;
            border: 1px solid #ccc;
            border-radius: var(--border-radius);
            box-sizing: border-box;
            transition: var(--transition);
            background-color: var(--card-bg);
            color: var(--text-color);
        }

        input[type="text"]:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 8px rgba(30, 136, 229, 0.3);
        }

        .search-icon {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #888;
        }

        .button-group {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            justify-content: center;
        }

        button {
            background-color: var(--primary-color);
            color: white;
            border: none;
            padding: 12px 30px;
            font-size: 16px;
            font-weight: 500;
            border-radius: var(--border-radius);
            cursor: pointer;
            transition: var(--transition);
        }

        button:hover {
            background-color: #1565c0;
            transform: translateY(-2px);
        }

        button.reset-btn {
            background-color: #95a5a6;
        }

        button.reset-btn:hover {
            background-color: #7f8c8d;
        }

        button.clear-btn {
            background-color: #e74c3c;
        }

        button.clear-btn:hover {
            background-color: #c0392b;
        }

        button.export-btn {
            background-color: #27ae60;
        }

        button.export-btn:hover {
            background-color: #219653;
        }

        button.print-btn {
            background-color: #8e44ad;
        }

        button.print-btn:hover {
            background-color: #7d3c98;
        }

        .add-code-form {
            background-color: var(--card-bg);
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            width: 100%;
            max-width: 600px;
            margin-bottom: 30px;
        }

        .add-code-form h3 {
            margin-top: 0;
            color: var(--primary-color);
        }

        .add-code-form .form-group {
            margin-bottom: 15px;
        }

        .add-code-form label {
            display: block;
            font-weight: 500;
            margin-bottom: 5px;
        }

        .add-code-form input {
            width: 100%;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: var(--border-radius);
            background-color: var(--card-bg);
            color: var(--text-color);
        }

        .message {
            text-align: center;
            margin-bottom: 20px;
            font-weight: 500;
        }

        .success {
            color: var(--success-color);
        }

        .error {
            color: var(--error-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            overflow: hidden;
            box-shadow: var(--shadow);
            margin-top: 20px;
            display: none;
        }

        table.visible {
            display: table;
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid rgba(0, 0, 0, 0.1);
        }

        th {
            background-color: var(--primary-color);
            color: white;
            font-weight: 600;
        }

        tr {
            transition: var(--transition);
        }

        tr:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }

        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
        }

        .pagination a {
            text-decoration: none;
            padding: 8px 16px;
            border-radius: var(--border-radius);
            background-color: var(--card-bg);
            color: var(--text-color);
            transition: var(--transition);
            border: 1px solid #ccc;
        }

        .pagination a:hover {
            background-color: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
        }

        .pagination a.active {
            background-color: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
            cursor: default;
        }

        .pagination a.disabled {
            color: #ccc;
            cursor: not-allowed;
            pointer-events: none;
        }

        .no-results {
            text-align: center;
            margin-top: 30px;
            font-size: 1.1rem;
            color: #666;
        }

        @media print {
            .navbar, .theme-toggle, .search-form, .add-code-form, .pagination, .button-group {
                display: none;
            }
            body {
                padding: 20px;
                background: white;
            }
            table {
                display: table !important;
                box-shadow: none;
            }
            .container {
                max-width: 100%;
            }
            h2 {
                text-align: center;
            }
        }

        @media (max-width: 768px) {
            .nav-menu {
                display: none;
                flex-direction: column;
                position: absolute;
                top: 60px;
                left: 0;
                right: 0;
                background-color: var(--card-bg);
                padding: 20px;
                box-shadow: var(--shadow);
            }

            .nav-menu.active {
                display: flex;
            }

            .hamburger {
                display: flex;
            }

            table, thead, tbody, th, td, tr {
                display: block;
            }

            thead {
                display: none;
            }

            tr {
                margin-bottom: 10px;
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
            }

            td {
                padding: 12px;
                position: relative;
                border: none;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }

            td::before {
                content: attr(data-label);
                font-weight: 600;
                color: var(--primary-color);
                flex: 1;
            }

            td::after {
                content: attr(data-value);
                flex: 2;
                text-align: right;
            }
        }

        @media (max-width: 480px) {
            h2 {
                font-size: 1.5rem;
            }

            input[type="text"], button {
                font-size: 14px;
            }

            .navbar-brand {
                font-size: 1.2rem;
            }

            .button-group {
                flex-direction: column;
                align-items: center;
            }

            .pagination a {
                padding: 6px 12px;
                font-size: 14px;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-container">
            <a href="/codes" class="navbar-brand">Postal Search</a>
            <button class="hamburger" onclick="toggleMenu()">
                <span></span>
                <span></span>
                <span></span>
            </button>
            <div class="nav-menu" id="nav-menu">
                <a href="/codes/find.php">Search Banks</a>
                <a href="/about">About</a>
                <a href="/contact">Contact</a>
            </div>
        </div>
    </nav>
    <button class="theme-toggle" onclick="toggleTheme()">🌙</button>
    <div class="container">
        <h2>Kenya Postal Code Search</h2>
        <?php if ($search): ?>
            <p class="search-info">Search results for: <span><?= htmlspecialchars($search) ?></span> (<?php echo $totalRecords; ?> results)</p>
        <?php endif; ?>
        <?php if ($success_message): ?>
            <p class="message success"><?= htmlspecialchars($success_message) ?></p>
        <?php endif; ?>
        <?php if ($error_message): ?>
            <p class="message error"><?= htmlspecialchars($error_message) ?></p>
        <?php endif; ?>
        <form method="GET" class="search-form">
            <div class="search-container">
                <input type="text" name="search" placeholder="Search by code, town, or county" 
                       value="<?= htmlspecialchars($search) ?>">
                <span class="search-icon">🔍</span>
            </div>
            <div class="button-group">
                <button type="submit">Search</button>
                <button type="button" class="reset-btn" onclick="resetSearch()">Reset</button>
                <button type="button" class="clear-btn" onclick="clearTable()">Clear Table</button>
            </div>
        </form>

        <form method="POST" class="add-code-form">
            <h3>Add New Postal Code</h3>
            <div class="form-group">
                <label for="code">Postal Code</label>
                <input type="text" name="code" id="code" placeholder="Enter postal code">
            </div>
            <div class="form-group">
                <label for="town">Town</label>
                <input type="text" name="town" id="town" placeholder="Enter town">
            </div>
            <div class="form-group">
                <label for="county">County</label>
                <input type="text" name="county" id="county" placeholder="Enter county">
            </div>
            <button type="submit" name="add_code">Add Postal Code</button>
        </form>

        <?php if ($results): ?>
            <form method="POST" class="button-group">
                <button type="submit" name="export" value="pdf" class="export-btn">Export PDF</button>
                <button type="submit" name="export" value="excel" class="export-btn">Export Excel</button>
                <button type="submit" name="export" value="word" class="export-btn">Export Word</button>
                <button type="submit" name="export" value="txt" class="export-btn">Export TXT</button>
                <button type="button" class="print-btn" onclick="window.print()">Print</button>
            </form>
            <table id="results-table" class="visible">
                <thead>
                    <tr>
                        <th>Postal Code</th>
                        <th>Town</th>
                        <th>County</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($results as $row): ?>
                    <tr>
                        <td data-label="Postal Code" data-value="<?= htmlspecialchars($row['code']) ?>">
                            <?= htmlspecialchars($row['code']) ?>
                        </td>
                        <td data-label="Town" data-value="<?= htmlspecialchars($row['town']) ?>">
                            <?= htmlspecialchars($row['town']) ?>
                        </td>
                        <td data-label="County" data-value="<?= htmlspecialchars($row['county']) ?>">
                            <?= htmlspecialchars($row['county']) ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php
                $queryString = $search ? "&search=" . urlencode($search) : "";
                ?>
                <a href="?page=1<?php echo $queryString; ?>" class="<?php echo $page <= 1 ? 'disabled' : ''; ?>">« First</a>
                <a href="?page=<?php echo $page - 1; ?><?php echo $queryString; ?>" class="<?php echo $page <= 1 ? 'disabled' : ''; ?>">‹ Prev</a>
                <?php
                $startPage = max(1, $page - 2);
                $endPage = min($totalPages, $page + 2);
                for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="?page=<?php echo $i; ?><?php echo $queryString; ?>" class="<?php echo $i === $page ? 'active' : ''; ?>"><?php echo $i; ?></a>
                <?php endfor; ?>
                <a href="?page=<?php echo $page + 1; ?><?php echo $queryString; ?>" class="<?php echo $page >= $totalPages ? 'disabled' : ''; ?>">Next ›</a>
                <a href="?page=<?php echo $totalPages; ?><?php echo $queryString; ?>" class="<?php echo $page >= $totalPages ? 'disabled' : ''; ?>">Last »</a>
            </div>
            <?php endif; ?>
        <?php else: ?>
            <p class="no-results">No results found.</p>
        <?php endif; ?>
    </div>

    <script>
        function toggleTheme() {
            const body = document.body;
            const currentTheme = body.getAttribute('data-theme');
            body.setAttribute('data-theme', currentTheme === 'dark' ? 'light' : 'dark');
            localStorage.setItem('theme', body.getAttribute('data-theme'));
        }

        function toggleMenu() {
            const menu = document.getElementById('nav-menu');
            menu.classList.toggle('active');
        }

        function resetSearch() {
            window.location.href = window.location.pathname;
        }

        function clearTable() {
            const table = document.getElementById('results-table');
            if (table) {
                table.classList.remove('visible');
            }
        }

        document.addEventListener('DOMContentLoaded', () => {
            const savedTheme = localStorage.getItem('theme') || 'light';
            document.body.setAttribute('data-theme', savedTheme);
        });
    </script>
</body>
</html>